
// GameView.java
	package com.example.zombieshooter;

	import android.content.Context;
	import android.graphics.Canvas;
	import android.graphics.Color;
	import android.graphics.Paint;
	import android.view.MotionEvent;
	import android.view.SurfaceHolder;
	import android.view.SurfaceView;
	import java.util.ArrayList;
	import java.util.Iterator;
	import java.util.Random;

	public class GameView extends SurfaceView implements Runnable {
		private Thread gameThread;
		private SurfaceHolder holder;
		private boolean isPlaying = true;
		private Paint paint;

		// Player
		private float playerX, playerY;
		private float playerSpeed = 10;
		private int playerSize = 40;

		// Control buttons
		private boolean moveUp, moveDown, moveLeft, moveRight;
		private int buttonSize = 150;

		// Bullets
		private ArrayList<Bullet> bullets;
		private long lastShootTime = 0;
		private int shootDelay = 300; // ms

		// Zombies
		private ArrayList<Zombie> zombies;
		private long lastZombieSpawn = 0;
		private int zombieSpawnDelay = 2000; // ms
		private Random random;

		// Score
		private int score = 0;

		// Screen size
		private int screenWidth, screenHeight;

		public GameView(Context context) {
			super(context);
			holder = getHolder();
			paint = new Paint();
			bullets = new ArrayList<>();
			zombies = new ArrayList<>();
			random = new Random();
		}

		@Override
		public void run() {
			while (isPlaying) {
				if (holder.getSurface().isValid()) {
					update();
					draw();
					sleep();
				}
			}
		}

		private void update() {
			// Get screen size
			if (screenWidth == 0) {
				screenWidth = getWidth();
				screenHeight = getHeight();
				playerX = screenWidth / 2;
				playerY = screenHeight - 200;
			}

			// Update player position
			if (moveUp && playerY > 0) playerY -= playerSpeed;
			if (moveDown && playerY < screenHeight - playerSize) playerY += playerSpeed;
			if (moveLeft && playerX > 0) playerX -= playerSpeed;
			if (moveRight && playerX < screenWidth - playerSize) playerX += playerSpeed;

			// Auto shoot
			long currentTime = System.currentTimeMillis();
			if (currentTime - lastShootTime > shootDelay) {
				bullets.add(new Bullet(playerX + playerSize / 2, playerY));
				lastShootTime = currentTime;
			}

			// Update bullets
			Iterator<Bullet> bulletIterator = bullets.iterator();
			while (bulletIterator.hasNext()) {
				Bullet bullet = bulletIterator.next();
				bullet.update();
				if (bullet.y < 0) {
					bulletIterator.remove();
				}
			}

			// Spawn zombies
			if (currentTime - lastZombieSpawn > zombieSpawnDelay) {
				int x = random.nextInt(screenWidth - 50);
				zombies.add(new Zombie(x, -50));
				lastZombieSpawn = currentTime;
			}

			// Update zombies
			Iterator<Zombie> zombieIterator = zombies.iterator();
			while (zombieIterator.hasNext()) {
				Zombie zombie = zombieIterator.next();
				zombie.update();

				// Check collision with bullets
				for (Bullet bullet : bullets) {
					if (isColliding(bullet.x, bullet.y, 10, zombie.x, zombie.y, zombie.size)) {
						zombieIterator.remove();
						bullets.remove(bullet);
						score += 10;
						break;
					}
				}

				// Check collision with player
				if (isColliding(playerX, playerY, playerSize, zombie.x, zombie.y, zombie.size)) {
					// Game over logic can be added here
					zombieIterator.remove();
					score -= 5;
				}

				// Remove if off screen
				if (zombie.y > screenHeight) {
					zombieIterator.remove();
				}
			}
		}

		private void draw() {
			Canvas canvas = holder.lockCanvas();
			if (canvas != null) {
				// Background
				canvas.drawColor(Color.rgb(20, 20, 30));

				// Draw player
				paint.setColor(Color.rgb(0, 200, 0));
				canvas.drawRect(playerX, playerY, playerX + playerSize, playerY + playerSize, paint);

				// Draw gun
				paint.setColor(Color.rgb(100, 100, 100));
				canvas.drawRect(playerX + 15, playerY - 20, playerX + 25, playerY, paint);

				// Draw bullets
				paint.setColor(Color.YELLOW);
				for (Bullet bullet : bullets) {
					canvas.drawCircle(bullet.x, bullet.y, 10, paint);
				}

				// Draw zombies
				for (Zombie zombie : zombies) {
					paint.setColor(Color.rgb(150, 0, 0));
					canvas.drawRect(zombie.x, zombie.y, zombie.x + zombie.size, zombie.y + zombie.size, paint);

					// Zombie eyes
					paint.setColor(Color.RED);
					canvas.drawCircle(zombie.x + 10, zombie.y + 10, 5, paint);
					canvas.drawCircle(zombie.x + 30, zombie.y + 10, 5, paint);
				}

				// Draw control buttons
				paint.setColor(Color.argb(100, 255, 255, 255));
				// Up button
				canvas.drawRect(70, screenHeight - 300, 70 + buttonSize, screenHeight - 300 + buttonSize, paint);
				// Down button
				canvas.drawRect(70, screenHeight - 140, 70 + buttonSize, screenHeight - 140 + buttonSize, paint);
				// Left button
				canvas.drawRect(0, screenHeight - 220, buttonSize, screenHeight - 220 + buttonSize, paint);
				// Right button
				canvas.drawRect(140, screenHeight - 220, 140 + buttonSize, screenHeight - 220 + buttonSize, paint);

				// Draw arrows
				paint.setColor(Color.WHITE);
				paint.setTextSize(60);
				canvas.drawText("↑", 110, screenHeight - 220, paint);
				canvas.drawText("↓", 110, screenHeight - 60, paint);
				canvas.drawText("←", 40, screenHeight - 140, paint);
				canvas.drawText("→", 180, screenHeight - 140, paint);

				// Draw score
				paint.setTextSize(50);
				canvas.drawText("Score: " + score, 50, 80, paint);

				holder.unlockCanvasAndPost(canvas);
			}
		}

		@Override
		public boolean onTouchEvent(MotionEvent event) {
			int action = event.getActionMasked();
			int pointerCount = event.getPointerCount();

			moveUp = moveDown = moveLeft = moveRight = false;

			for (int i = 0; i < pointerCount; i++) {
				float x = event.getX(i);
				float y = event.getY(i);

				if (action == MotionEvent.ACTION_DOWN || action == MotionEvent.ACTION_POINTER_DOWN || action == MotionEvent.ACTION_MOVE) {
					// Up button
					if (x >= 70 && x <= 70 + buttonSize && y >= screenHeight - 300 && y <= screenHeight - 300 + buttonSize) {
						moveUp = true;
					}
					// Down button
					if (x >= 70 && x <= 70 + buttonSize && y >= screenHeight - 140 && y <= screenHeight - 140 + buttonSize) {
						moveDown = true;
					}
					// Left button
					if (x >= 0 && x <= buttonSize && y >= screenHeight - 220 && y <= screenHeight - 220 + buttonSize) {
						moveLeft = true;
					}
					// Right button
					if (x >= 140 && x <= 140 + buttonSize && y >= screenHeight - 220 && y <= screenHeight - 220 + buttonSize) {
						moveRight = true;
					}
				}
			}

			return true;
		}

		private boolean isColliding(float x1, float y1, float size1, float x2, float y2, float size2) {
			return x1 < x2 + size2 && x1 + size1 > x2 && y1 < y2 + size2 && y1 + size1 > y2;
		}

		private void sleep() {
			try {
				Thread.sleep(17); // ~60 FPS
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}

		public void resume() {
			isPlaying = true;
			gameThread = new Thread(this);
			gameThread.start();
		}

		public void pause() {
			try {
				isPlaying = false;
				gameThread.join();
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}

		// Bullet class
		class Bullet {
			float x, y;
			float speed = 20;

			Bullet(float x, float y) {
				this.x = x;
				this.y = y;
			}

			void update() {
				y -= speed;
			}
		}

		// Zombie class
		class Zombie {
			float x, y;
			int size = 50;
			float speed = 3 + random.nextFloat() * 2;

			Zombie(float x, float y) {
				this.x = x;
				this.y = y;
			}

			void update() {
				y += speed;
			}
		}
	}
